'use client';

import { useMemo, useEffect } from 'react';
import Link from 'next/link';
import {
  useCollection,
  useFirestore,
  useMemoFirebase,
  useAuth,
  initiateAnonymousSignIn,
  useUser,
} from '@/firebase';
import { collection, query, orderBy } from 'firebase/firestore';
import type { Order, OrderItem } from '@/lib/types';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from '@/components/ui/accordion';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { ArrowLeft, ListOrdered } from 'lucide-react';
import AdminNav from '@/components/admin-nav';

const formatPrice = (price: number) => {
  return price.toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
};

const formatDate = (date: Date) => {
  return date.toLocaleString('pt-BR', {
    day: '2-digit',
    month: '2-digit',
    year: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
  });
};

const formatOrderItem = (item: OrderItem) => {
  let name = `${item.quantity}x ${item.productName}`;
  if(item.addons && item.addons.length > 0) {
    name += ` (Adicionais: ${item.addons.map(a => a.name).join(', ')})`
  }
  return name;
}

export default function AdminOrdersPage() {
  const firestore = useFirestore();
  const auth = useAuth();
  const { user, isUserLoading } = useUser();

  useEffect(() => {
    // Sign in anonymously if there's no user and auth is ready
    if (!user && !isUserLoading && auth) {
      initiateAnonymousSignIn(auth);
    }
  }, [user, isUserLoading, auth]);


  const ordersQuery = useMemoFirebase(
    () =>
      firestore && user // Only create query if user is authenticated
        ? query(collection(firestore, 'orders'), orderBy('orderDate', 'desc'))
        : null,
    [firestore, user]
  );

  const {
    data: orders,
    isLoading,
    error,
  } = useCollection<Order>(ordersQuery);

  const ordersWithDate = useMemo(() => {
    return orders?.map((order) => ({
      ...order,
      // Firestore timestamp to JS Date
      orderDate: (order.orderDate as any).toDate(),
    }));
  }, [orders]);
  
  const showLoading = isLoading || isUserLoading;

  return (
    <div className="max-w-7xl mx-auto p-4 md:p-6">
      <div className="flex items-center gap-4 mb-6">
        <Link href="/admin/products">
          <Button variant="outline" size="icon">
            <ArrowLeft className="h-4 w-4" />
          </Button>
        </Link>
        <h1 className="text-3xl font-bold">Gerenciar Pedidos</h1>
      </div>
      
      <AdminNav />

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ListOrdered className="h-6 w-6" />
            Pedidos Recebidos
          </CardTitle>
          <CardDescription>
            Visualize os pedidos dos clientes em tempo real.
          </CardDescription>
        </CardHeader>
        <CardContent>
          {showLoading && <p>Carregando pedidos...</p>}
          {error && (
            <p className="text-destructive">
              Ocorreu um erro ao carregar os pedidos: {error.message}
            </p>
          )}
          {!showLoading && !user && (
             <p className="text-muted-foreground">Autenticando para buscar pedidos...</p>
          )}
          {!showLoading && user && !ordersWithDate?.length && (
            <p className="text-muted-foreground">Nenhum pedido encontrado.</p>
          )}
          {ordersWithDate && ordersWithDate.length > 0 && (
            <Accordion type="multiple" className="space-y-4">
              {ordersWithDate.map((order) => (
                <AccordionItem
                  key={order.id}
                  value={order.id}
                  className="border rounded-lg"
                >
                  <AccordionTrigger className="p-4 hover:no-underline">
                    <div className="flex justify-between items-center w-full">
                      <div className="text-left">
                        <p className="font-bold">{order.customerName}</p>
                        <p className="text-sm text-muted-foreground">
                          {formatDate(order.orderDate)}
                        </p>
                      </div>
                      <div className="text-right">
                         <p className="font-bold text-lg text-primary">{formatPrice(order.totalAmount)}</p>
                         <Badge>{order.paymentMethod}</Badge>
                      </div>
                    </div>
                  </AccordionTrigger>
                  <AccordionContent className="p-4 border-t">
                    <div className="space-y-4">
                        <div>
                            <h4 className='font-semibold'>Endereço de Entrega</h4>
                            <p className='text-muted-foreground'>{order.deliveryAddress}</p>
                        </div>
                        <Separator />
                        <div>
                            <h4 className='font-semibold mb-2'>Itens do Pedido</h4>
                            <ul className="space-y-2">
                            {(order.orderItems || []).map((item, index) => (
                                <li key={index} className="flex justify-between">
                                    <span>{formatOrderItem(item)}</span>
                                    <span>{formatPrice(item.unitPrice * item.quantity)}</span>
                                </li>
                            ))}
                            </ul>
                        </div>
                    </div>
                  </AccordionContent>
                </AccordionItem>
              ))}
            </Accordion>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
